// 引入 Electron 相关模块
const { app, BrowserWindow, Menu, ipcMain, shell, dialog } = require('electron');
const path = require('path');
const fs = require('fs');  // 文件系统模块
const crypto = require('crypto');  // 引入加密块


// 定义主窗口变量
let mainWindow;
let loadingWindow;

// 获取编辑器跟目录
let rootPath = path.dirname(app.getPath('exe'));

// 获取APP根目录
const appPath = app.getAppPath();

app.setPath('userData', rootPath + '/temp');

// 创建窗口的函数
function createWindow() {

    // 创建加载窗口
    loadingWindow = new BrowserWindow({
        width: 480,
        height: 180,
        frame: false,
        resizable: false,
        show: false,
        webPreferences: {
            nodeIntegration: true,
            contextIsolation: false
        }
    });

    // 加载加载页面
    loadingWindow.loadFile('loading.html');

    // 当加载页面准备好显示时显示加载窗口
    loadingWindow.once('ready-to-show', () => {
        loadingWindow.show();
    });

    // 创建主窗口
    mainWindow = new BrowserWindow({
        width: 1280,
        height: 720,
        frame: false, // 隐藏原生标题栏
        show: false,
        webPreferences: {
            nodeIntegration: false, // 不允许在渲染进程中直接使用 Node.js API，增强安全性
            contextIsolation: true, // 开启上下文隔离，防止渲染进程直接访问主进程全局对象
            preload: path.join(appPath, 'preload.js'), // 指定预加载脚本的路径
            devTools: false  // 禁用开发者工具
        }
    });

    // 主窗口加载远程页面
    mainWindow.loadURL('https://www.gameauto.cn/editor/main');
    // mainWindow.loadURL('http://localhost/editor/main');

    // 当主窗口加载完成时，隐藏加载窗口并显示主窗口
    mainWindow.webContents.on('did-finish-load', () => {
        if (loadingWindow) {
            loadingWindow.close();
            loadingWindow = null;
        }
        mainWindow.show();
        //mainWindow.maximize();  // 设置为最大化
    });

    // 开启开发者工具
    mainWindow.webContents.openDevTools();
    // 移除默认的菜单栏
    Menu.setApplicationMenu(null);
    // 设置窗口的最小缩放尺寸，宽度最小为 1280，高度最小为 720
    mainWindow.setMinimumSize(1280, 720);
    // 监听窗口关闭事件，当窗口关闭时将主窗口变量置为 null
    mainWindow.on('closed', () => {
        mainWindow = null;
    });
    // 监听窗口最大化事件，当窗口最大化时向渲染进程发送消息
    mainWindow.on('maximize', () => {
        mainWindow.webContents.send('window-maximized');
    });
    // 监听窗口从最大化恢复事件，当窗口从最大化恢复时向渲染进程发送消息
    mainWindow.on('unmaximize', () => {
        mainWindow.webContents.send('window-unmaximized');
    });
}


// 当 Electron 应用准备好时，执行创建窗口的操作
app.whenReady().then(() => {
    createWindow();
    // 监听应用激活事件，当应用被激活且没有打开的窗口时，创建新窗口
    app.on('activate', () => {
        if (BrowserWindow.getAllWindows().length === 0) createWindow();
    });
});


// 监听所有窗口关闭事件，当所有窗口关闭时（除了 macOS 系统），退出应用
app.on('window-all-closed', () => {
    if (process.platform!== 'darwin') app.quit();
});


// 监听来自渲染进程的最小化窗口消息，当接收到消息时将窗口最小化
ipcMain.on('minimize-window', () => {
    mainWindow.minimize();
});


// 监听来自渲染进程的切换窗口最大化状态消息，根据当前窗口状态进行最大化或恢复操作
ipcMain.on('toggle-maximize-window', () => {
    if (mainWindow.isMaximized()) {
        mainWindow.unmaximize();
    } else {
        mainWindow.maximize();
    }
});


// 监听来自渲染进程的关闭窗口消息，当接收到消息时关闭窗口
ipcMain.on('close-window', () => {
    mainWindow.close();
});


// 递归复制文件夹
function copyFolderRecursive(source, target) {
    if (!fs.existsSync(target)) {
        fs.mkdirSync(target);
    }
    const files = fs.readdirSync(source);
    for (const file of files) {
        const sourcePath = path.join(source, file);
        const targetPath = path.join(target, file);
        const stat = fs.statSync(sourcePath);
        if (stat.isDirectory()) {
            copyFolderRecursive(sourcePath, targetPath);
        } else {
            fs.copyFileSync(sourcePath, targetPath);
        }
    }
}


// 递归删除文件夹及其子文件夹和文件
function deleteFolderRecursive(folderPath) {
    if (fs.existsSync(folderPath)) {
        fs.readdirSync(folderPath).forEach((file) => {
            const curPath = path.join(folderPath, file);
            if (fs.lstatSync(curPath).isDirectory()) {
                // 递归删除子文件夹
                deleteFolderRecursive(curPath);
            } else {
                // 删除文件
                fs.unlinkSync(curPath);
            }
        });
        // 删除空文件夹
        fs.rmdirSync(folderPath);
    }
}


// 更新本地项目文件和数据
ipcMain.handle('update-local', (event, ProjectId, ProjectData, ProjectLogic) => {
    try {
        // 检查本地是否有项目文件夹，如果没有则创建
        if (!fs.existsSync(path.join(rootPath + '\\project\\' + ProjectId))) {
            // 复制脚本母版到项目文件夹
            copyFolderRecursive(rootPath + "\\source", rootPath + "\\project\\" + ProjectId);
        }
        // 更新library.json
        const filePath = path.join(rootPath + '\\project\\' + ProjectId + '\\library.json');
        fs.readFile(filePath, 'utf8', (err, data) => {
            // 解析并修改 JSON 数据
            const jsonData = JSON.parse(data);
            jsonData.Project = ProjectData;
            jsonData.Logic = ProjectLogic;
            // 将修改后的JSON写回文件
            fs.writeFile(filePath, JSON.stringify(jsonData, null, 2), 'utf8', (err) => {});
        });
    } catch (err) {
        return false;
    }
});




// 删除项目
ipcMain.handle('project-delete', (event, id) => {
    try {
        deleteFolderRecursive(rootPath + "\\project\\" + id);
        deleteFolderRecursive(rootPath + "\\project\\" + id + ".json");
    } catch (err) {
        return false;
    }
});

// 打开项目根目录
ipcMain.handle('open-project-folder', (event, CurrentProjectFolder) => {
    try {
        shell.openPath(rootPath + "\\project\\" + CurrentProjectFolder).then((error) => {});
    } catch (err) {
        return false;
    }
});

// 打开图片素材目录
ipcMain.handle('open-project-folder-img', (event, CurrentProjectFolder) => {
    try {
        shell.openPath(rootPath + "\\project\\" + CurrentProjectFolder + "\\img").then((error) => {}); 
    } catch (err) {
        return false;
    }
});

// 打开网页
ipcMain.handle('open-url', (event, url) => {
    shell.openExternal(url);
});


// 确认弹框
ipcMain.handle('confirm-box', (event, title, text) => {
    return dialog.showMessageBox({
        type: 'warning',
        buttons: ['确认', '取消'],
        title: title,
        message: text || '你确定要执行此操作吗？',
        cancelId: 1
    }).then(result => {
        if (result.response === 0) {
            return true;
        } else {
            return false;
        }
    });
});


// 读取项目的功能逻辑的JSON文件，并返回给渲染进程
ipcMain.handle('get-logic-editor-json', (event, CurrentProjectId) => {
    try {
        // 判断本地是否存在相应的文件，如果没有则创建
        const filePath = path.join(rootPath + '\\project\\' + CurrentProjectId + '.json')
        if (!fs.existsSync(filePath)) {
            fs.writeFile(filePath, JSON.stringify({}, null, 2), (err) => {});
        }
        const luaFilePath = path.join(rootPath + '\\project\\' + CurrentProjectId + '\\lua.lua')
        if (!fs.existsSync(luaFilePath)) {
            fs.writeFile(luaFilePath, JSON.stringify({}, null, 2), (err) => {});
        }
        // 读取并返回文件内容
        const data = fs.readFileSync(filePath, 'utf8');
        const jsonData = JSON.parse(data);
        return jsonData;
    } catch (err) {
        return false;
    }
});


// 保存功能逻辑代码到相应的文件里
ipcMain.handle('save-logic', (event, CurrentProjectId, jsonData, luaData) => {
    try {
        // 写入 JSON 文件
        fs.writeFile(path.join(rootPath, '\\project\\' + CurrentProjectId + '.json'), JSON.stringify(jsonData, null, 2), 'utf8', (err) => {});

        // // 生成随机IV
        // const iv = crypto.randomBytes(16);
        // // 创建加密器
        // const cipher = crypto.createCipheriv('aes-256-cbc', "XJ-Wzy"+CurrentProjectId+"4e/m-Lw+XJ", iv);
        // // 加密数据
        // let encrypted = cipher.update(luaData, 'utf8', 'hex');
        // encrypted += cipher.final('hex');
        // // 将IV和密文拼接（IV在前，密文在后）
        // luaData = iv.toString('hex') + encrypted;
  
        // 写入 LUA 文件
        fs.writeFile(path.join(rootPath, '\\project\\' + CurrentProjectId + '\\lua.lua'), luaData, (err) => {});
    } catch (err) {
        return false;
    }
});
